import { world } from "@minecraft/server";
import { ModalFormData } from "@minecraft/server-ui";
import { generate } from "../generator/jigsaw_generator";
import { settings } from "../../settings";
class OnInteractComponent {
    async onPlayerInteract(event) {
        const dimension = event.dimension;
        const block = event.block;
        const player = event.player;
        const jigsawEntity = this.findJigsawEntity(dimension, block.location);
        if (!jigsawEntity) return;
        const jigsawData = JSON.parse(jigsawEntity.getDynamicProperty("jigsawData"));
        const form = this.createJigsawForm(jigsawData);
        const formResponse = await form.show(player);
        this.finalizeForm(jigsawEntity, jigsawData, formResponse);
    }
    findJigsawEntity(dimension, location) {
        const entities = dimension.getEntitiesAtBlockLocation(location);
        return entities.find((entity)=>entity.typeId == 'jigsaw:jigsaw_data' && typeof entity.getDynamicProperty('jigsawData') == 'string');
    }
    createJigsawForm(jigsawData) {
        const form = new ModalFormData().title("Jigsaw Block Data");
        form.textField("Target pool:", "minecraft:empty", jigsawData.targetPool);
        form.textField("Name:", "minecraft:empty", jigsawData.name);
        form.textField("Target name:", "minecraft:empty", jigsawData.targetName);
        form.textField("Turns into:", "minecraft:air", jigsawData.turnsInto);
        form.textField("Selection Priority", "0", jigsawData.selectionPriority.toString());
        form.textField("Placement Priority", "0", jigsawData.placementPriority.toString());
        if (jigsawData.blockFace === "up" || jigsawData.blockFace === "down") {
            form.dropdown("Joint type:", [
                "rollable",
                "aligned"
            ], jigsawData.jointType == "rollable" ? 0 : 1);
        }
        form.toggle("Generate", false);
        const incrementAmount = Math.ceil(settings.jigsawMaxLevels / 20) == 0 ? 1 : Math.ceil(settings.jigsawMaxLevels / 20);
        form.slider("Levels", 0, settings.jigsawMaxLevels, incrementAmount, 7);
        return form;
    }
    finalizeForm(jigsawEntity, jigsawData, response) {
        let shouldGenerate = false;
        if (response.canceled) return;
        jigsawData.targetPool = response.formValues[0].toString();
        jigsawData.name = response.formValues[1].toString();
        jigsawData.targetName = response.formValues[2].toString();
        jigsawData.turnsInto = response.formValues[3].toString();
        let selectionPriority = parseInt(response.formValues[4].toString());
        if (isNaN(selectionPriority)) selectionPriority = 0;
        jigsawData.selectionPriority = selectionPriority;
        let placementPriority = parseInt(response.formValues[5].toString());
        if (isNaN(placementPriority)) placementPriority = 0;
        jigsawData.placementPriority = placementPriority;
        if (jigsawData.blockFace === "up" || jigsawData.blockFace === "down") {
            if (response.formValues[6] === 0) jigsawData.jointType = "rollable";
            else jigsawData.jointType = "aligned";
            shouldGenerate = response.formValues[7];
            if (shouldGenerate) jigsawData.levels = response.formValues[7];
        } else {
            shouldGenerate = response.formValues[6];
            if (shouldGenerate) jigsawData.levels = response.formValues[7];
        }
        jigsawEntity.setDynamicProperty("jigsawData", JSON.stringify(jigsawData, null, 4));
        if (shouldGenerate) generate(jigsawEntity);
    }
    constructor(){
        this.onPlayerInteract = this.onPlayerInteract.bind(this);
    }
}
world.beforeEvents.worldInitialize.subscribe((event)=>{
    event.blockTypeRegistry.registerCustomComponent('jigsaw:on_interact', new OnInteractComponent());
});
